# Chapter 8: Testing for Confidence: Modern Strategies and Tools

# Chapter 8: Testing for Confidence — Modern Strategies and Tools

## Introduction: Why Testing is Your Application’s Safety Net

Imagine building a skyscraper without inspecting the beams. Risky, right? In software, testing is your inspection—catching cracks before they become disasters. As React and TypeScript evolve, so must your testing approach.

Modern React 19 apps use server-first patterns, TypeScript, and new tools. Testing is now critical for speed, safety, and business value. This chapter shows how to use Vitest, React Testing Library, Mock Service Worker (MSW), and Playwright to build confidence—layer by layer.

You'll learn:

- The modern testing pyramid: unit, integration, E2E
- Migrating to Vitest for fast, reliable tests
- User-centric testing with React Testing Library
- Mocking APIs with MSW
- Automating real user flows with Playwright
- Measuring and maintaining test coverage

Testing is your safety net. Let’s see how to build it right.

---

## The Modern Testing Pyramid: From Unit to E2E

Think of testing like car manufacturing. You check each part (unit), how parts fit (integration), and take the car for a spin (E2E).

**Testing pyramid:**

- **Unit tests:** Fast, check small pieces in isolation.
- **Integration tests:** Check how components work together.
- **E2E tests:** Simulate real user journeys in the browser.

A balanced mix gives speed, coverage, and confidence. Let’s break down each layer.

---

### Unit Testing: The Foundation

Unit tests are your first defense. They check small, isolated pieces—like utility functions or pure components. They run fast and catch bugs early.

---

### Integration Testing: Checking the Connections

Integration tests check how parts connect. In React, this means testing component rendering, form handling, or hooks with real data. They find bugs in how modules work together.

---

### End-to-End Testing: The Full User Experience

E2E tests automate real user flows—like adding to cart or logging in. Tools like Playwright run these in real browsers, catching issues that only show up in production-like environments.

---

## Unit and Component Testing with Vitest

Vitest is a modern test runner—fast, zero-config, and TypeScript-native. React Testing Library (RTL) helps you test components as users interact with them, not by poking at internals.

### Migrating from Jest to Vitest

Migrating is simple. Update configs, adjust mocks, and most tests run unchanged. Vitest is much faster, especially in watch mode.

#### Migrating a Basic Test from Jest to Vitest

Before running your tests, update configs as shown below.

### `jest.config.js` and `vitest.config.ts` — Migrating Configs

```typescript

// jest.config.js
module.exports = { testEnvironment: 'jsdom' };

// vitest.config.ts
import { defineConfig } from 'vitest/config';
export default defineConfig({
  test: { environment: 'jsdom' }
});

```

- `jest.config.js` sets up Jest.
- `vitest.config.ts` sets up Vitest with jsdom for DOM testing.
- No transpilation needed with Node.js v22+ and TypeScript 5.8+.

---

### Writing User-Centric Component Tests

Test what users see and do—not internals. RTL focuses on user actions and visible text.

### `Counter.test.tsx` — Testing a Button Click

Before the code: This test ensures clicking the "Increment" button updates the count.

```typescript

import { render, screen, fireEvent }
  from '@testing-library/react';
import { describe, it, expect } from 'vitest';
import { Counter } from './Counter';

describe('Counter', () => {
  it('increments count on click', () => {
    render(<Counter />);
    fireEvent.click(screen.getByText('Increment'));
    expect(
      screen.getByText('Count: 1')
    ).toBeInTheDocument();
  });
});

```

- Renders the `Counter` component.
- Simulates a click on "Increment".
- Checks that the displayed count updates.

---

### Testing State Logic

Test custom hooks or reducers in isolation to ensure business logic is solid.

### `useCart.test.ts` — Testing a Custom React Hook

Before the code: This test checks that adding an item updates the cart state.

```typescript

import { renderHook, act }
  from '@testing-library/react';
import { useCart } from './useCart';

describe('useCart', () => {
  it('adds an item to the cart', () => {
    const { result } = renderHook(() => useCart());
    act(() => {
      result.current.addItem({
        id: 'prod-1',
        name: 'Widget'
      });
    });
    expect(result.current.items).toHaveLength(1);
  });
});

```

- Mounts the `useCart` hook.
- Calls `addItem` to add a product.
- Checks that the cart now has one item.

---

## Mocking and Service Workers: Simulating the Real World

Testing against real APIs is slow and risky. Mock Service Worker (MSW) lets you intercept API calls and return fake data—fast and safe.

### Setting Up Mock Service Worker

Install MSW and define handlers for your API endpoints.

### `handlers.ts` — Basic MSW Handler Setup

Before the code: This handler mocks a GET request to `/api/products`.

```typescript

import { http, HttpResponse } from "msw";

interface Product {
  id: string;
  name: string;
}

export const handlers = [
  http.get<never, never, Product[]>(
    "/api/products",
    () => {
      return HttpResponse.json([
        { id: "1", name: "Widget" }
      ]);
    }
  )
];

```

- Defines a mock for GET `/api/products`.
- Returns a static product list.

---

### `server.ts` — MSW Test Server Setup

Before the code: Set up the MSW server for use in tests.

```typescript

import { setupServer } from "msw/node";
import { handlers } from "./handlers";

export const server = setupServer(...handlers);

```

- Sets up the mock server with handlers.

---

### `setupTests.ts` — MSW Lifecycle Hooks for Vitest

Before the code: Ensures MSW starts and resets for each test.

```typescript

import { server } from "./mocks/server";

beforeAll(() => server.listen());
afterEach(() => server.resetHandlers());
afterAll(() => server.close());

```

- Starts MSW before tests.
- Resets handlers after each test.
- Stops MSW after all tests.

---

### Testing Server-Client Boundaries and Actions

Use MSW to mock server responses for Actions—functions that mutate data.

### `CheckoutForm.test.tsx` — Testing a Server Action with MSW

Before the code: This test simulates a successful checkout and checks for confirmation.

```typescript

import { render, screen, fireEvent, waitFor }
  from "@testing-library/react";
import { CheckoutForm } from "./CheckoutForm";

it("handles successful checkout", async () => {
  render(<CheckoutForm />);
  fireEvent.change(
    screen.getByLabelText(/name/i),
    { target: { value: "Alice" } }
  );
  fireEvent.change(
    screen.getByLabelText(/card/i),
    { target: { value: "4242 4242 4242 4242" } }
  );
  fireEvent.click(screen.getByText("Submit"));
  await waitFor(() => {
    expect(
      screen.getByText(/Order confirmed/i)
    ).toBeInTheDocument();
  });
});

```

- Fills out form fields.
- Clicks submit.
- Waits for confirmation message.

---

### Simulating an Error Response with MSW

Override handlers to test error scenarios.

### `CheckoutFormError.test.tsx` — Simulating Error Response

Before the code: This test checks for error handling on checkout failure.

```typescript

import { http, HttpResponse } from "msw";
import { server } from "./mocks/server";
import { render, screen, fireEvent, waitFor }
  from "@testing-library/react";
import { CheckoutForm } from "./CheckoutForm";

server.use(
  http.post<never, { name: string; card: string }, { error: string }>(
    "/api/checkout",
    () => {
      return HttpResponse.json(
        { error: "Payment failed" },
        { status: 500 }
      );
    }
  )
);

it("shows error on failed checkout", async () => {
  render(<CheckoutForm />);
  // Simulate form fill and submit...
  await waitFor(() => {
    expect(
      screen.getByText(/Payment failed/i)
    ).toBeInTheDocument();
  });
});

```

- Overrides the handler to return an error.
- Checks that the error message appears.

---

## End-to-End Testing with Playwright

E2E tests automate user flows in real browsers. Playwright makes this easy, reliable, and cross-browser.

### Setting Up Playwright

Install Playwright and its browsers.

### Playwright Installation

Before the code: Install Playwright and browsers.

```sh

npm install --save-dev @playwright/test
npx playwright install

```

- Installs Playwright test runner.
- Downloads Chromium, Firefox, and WebKit.

---

### `playwright.config.ts` — Sample Playwright Configuration

Before the code: Configure Playwright for parallel, cross-browser E2E tests.

```typescript

import { defineConfig, devices } from '@playwright/test';

export default defineConfig({
  testDir: './e2e',
  timeout: 30000,
  retries: 1,
  fullyParallel: true,
  use: {
    baseURL: 'http://localhost:3000',
    trace: 'on-first-retry',
    screenshot: 'only-on-failure',
    video: 'retain-on-failure'
  },
  projects: [
    { name: 'chromium', use: { ...devices['Desktop Chrome'] } },
    { name: 'firefox', use: { ...devices['Desktop Firefox'] } },
    { name: 'webkit', use: { ...devices['Desktop Safari'] } }
  ],
  reporter: [['html', { open: 'never' }], ['list']]
});

```

- Runs tests in all browsers.
- Collects traces, screenshots, and videos on failure.

---

### Automating Complex User Flows

Write E2E tests to simulate real journeys.

### `login.e2e.ts` — Playwright Test Example: Login Flow

Before the code: This test checks the login process end-to-end.

```typescript

import { test, expect } from '@playwright/test';

test('user can log in', async ({ page }) => {
  await page.goto('http://localhost:3000/login');
  await page.fill('input[name="email"]',
    'user@example.com');
  await page.fill('input[name="password"]',
    'securepassword');
  await page.click('button[type="submit"]');
  await expect(page).toHaveURL(
    'http://localhost:3000/dashboard'
  );
});

```

- Opens login page.
- Fills in credentials.
- Submits and checks redirect.

---

### CI Integration for Continuous Quality

Automate E2E tests in CI to catch regressions early.

### `e2e.yml` — GitHub Actions Workflow for Playwright

Before the code: This workflow runs Playwright tests on every push or PR.

```yaml

name: E2E Tests
on: [push, pull_request]
jobs:
  test:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      - uses: actions/setup-node@v4
        with:
          node-version: '20'
      - run: npm ci
      - run: npx playwright install --with-deps
      - run: npm run test:e2e

```

- Runs on every push/PR.
- Installs dependencies and browsers.
- Runs all E2E tests.

---

## Test Coverage, Reporting, and Maintenance

Coverage tells you which code is tested. High coverage reduces risk but focus on critical paths.

### Generating Coverage with Vitest

Before the code: Run this command to generate a coverage report.

```sh

vitest run --coverage

```

- Outputs a summary.
- View full report in `coverage/index.html`.

---

### Maintaining Tests Over Time

Keep tests updated as code changes. Refactor tests with code, remove obsolete ones, and use clear names. Healthy tests mean fewer bugs and happier teams.

---

## Summary and Key Takeaways

Testing is your app’s safety net. Use the modern pyramid: many fast unit tests, some integration tests, and a few E2E tests. Vitest, React Testing Library, MSW, and Playwright are your core tools.

- **Vitest:** Fast, TypeScript-friendly test runner.
- **React Testing Library:** User-focused component tests.
- **MSW:** Reliable API mocking.
- **Playwright:** Real browser E2E automation.

Maintain tests as your code evolves. Focus on business-critical paths. Automate tests in CI to catch issues before users do.

For more on state management, see Chapter 6. For DevOps and deployment, see Chapter 10.

---

## Key Ideas and Glossary

| Key Idea                                   | Definition                                                                 |
|---------------------------------------------|----------------------------------------------------------------------------|
| Unit Test                                  | Test a single function/component in isolation                              |
| Integration Test                           | Test how components/modules work together                                  |
| End-to-End (E2E) Test                      | Simulate real user flows in the full app                                   |
| Mock Service Worker (MSW)                   | Tool for intercepting and mocking network requests in tests                |
| Coverage                                   | Percentage of codebase exercised by tests                                  |
| Playwright                                 | E2E browser automation tool                                                |
| React Testing Library (RTL)                 | Library for user-centric component testing                                 |
| Action (React 19)                          | Server-side mutation triggered from the client                             |
| Server Component (React 19)                 | Component rendered on the server, not in the browser                       |
| Accessibility Testing (axe-core)            | Automated checks for accessibility compliance                              |

---

## Exercises and Next Steps

1. **Migrate a Jest test to Vitest.**  
   - Update configs, run both, and compare speed.

2. **Write a user-centric test for a login form.**  
   - Use React Testing Library and Vitest. Cover both success and failure with MSW.

3. **Set up MSW to mock an API endpoint.**  
   - Test both success and error responses in your component.

4. **Create a Playwright E2E test for cart checkout.**  
   - Run it in at least two browsers.

5. **Generate a coverage report with Vitest.**  
   - Find an uncovered path and write a test for it.

For deeper dives, see Chapter 8 for advanced mocking and E2E strategies, Chapter 6 for state logic, and Chapter 10 for CI/CD integration.