# Chapter 7: Full-Stack Type Safety and API Integration: tRPC, OpenAPI, and Authentication

# Full-Stack Type Safety and API Integration: tRPC, OpenAPI, and Authentication

## Introduction: Type Safety as the Backbone of Modern Web APIs

Imagine your business as a global shipping company. Packages (data) cross borders (APIs) daily. If the paperwork (types) is wrong, shipments are delayed or lost. In software, type mismatches are these lost packages—except they crash your app, frustrate users, and waste developer time.

Type safety is your customs office. It verifies that every data package matches expectations before crossing boundaries. With tools like tRPC and OpenAPI, you can enforce these contracts, prevent bugs, and accelerate development. 

In this chapter, you’ll learn:
- Why type-safe APIs are essential for reliability and speed
- How to use tRPC for TypeScript monorepos and OpenAPI for polyglot teams
- How to organize shared types in a monorepo
- How to automate contract checks in CI
- How to secure your APIs with robust authentication and authorization patterns

Let’s turn API chaos into a well-oiled machine.

---

## The Case for Type-Safe APIs

APIs are business contracts. Both sides must agree on what’s delivered. Type safety is the legal language that ensures this. Without it, your frontend might expect a number for `price`, but the backend sends a string—leading to runtime errors.

### Business Analogy: Contracts in Supply Chains

A shipping contract specifies contents and delivery. If ignored, things go missing. APIs work the same way. Clear, enforced contracts prevent confusion and blame.

### Eliminating Entire Classes of Runtime Errors

A single type mismatch can break your app. Type safety catches these at compile time.

#### ### Type Mismatch Example: The Bug You’ll Never See Again

This example shows how a backend sending the wrong type can break the frontend.

### Type Mismatch Example: The Bug You’ll Never See Again (`displayPrice.ts`)

Before the code, note: The backend sends `price` as a string, but the frontend expects a number. This causes a runtime error.

```typescript

// Backend sends this JSON:
{
  "price": "19.99"
}

// Frontend expects a number:
function displayPrice(product: { price: number }) {
  // Formats the price as currency
  return `$${product.price.toFixed(2)}`;
}
// Without type safety, this fails at runtime!
// With shared types, it fails at compile time.
```

- Backend sends `price` as a string.
- Frontend expects `price` as a number.
- Calling `.toFixed(2)` on a string throws at runtime.
- Type safety catches this during development, not production.

### Developer Productivity and Onboarding Benefits

Shared types are living documentation. New developers get autocompletion and instant feedback. No more guessing what an endpoint returns.

Modern tools like tRPC and OpenAPI make this practical—even for large teams. Type safety is not just technical; it’s a business advantage.

---

## tRPC: TypeScript Monorepo Integration

tRPC is a direct hotline between frontend and backend. Define APIs in TypeScript. Use them everywhere with enforced types.

A monorepo keeps shared types and utilities in one place. Turborepo or pnpm workspaces help manage this at scale.

### Setting Up tRPC for Instant Type-Safe APIs

Start with a monorepo:

### Monorepo Directory Structure (Simplified)

```plaintext

/apps
  /backend
  /frontend
/packages
  /types
```

Install dependencies:

### Installing tRPC and Dependencies (2025 Edition)

```sh

pnpm add @trpc/server zod
pnpm add @trpc/client @trpc/react-query @tanstack/react-query
```

Define your first tRPC router.

### Basic tRPC Router Definition (`productRouter.ts`)

This code defines a product API with full type safety using TypeScript and Zod.

```typescript

import { initTRPC } from '@trpc/server';
import { z } from 'zod';
import { getProducts, getProductById } from '../services/productService';
import { Product } from '@your-org/types';

const t = initTRPC.create();

export const productRouter = t.router({
  // Returns Product[]
  getAll: t.procedure.query((): Product[] => getProducts()),
  // Takes product ID, returns Product | undefined
  getById: t.procedure.input(z.string()).query(
    ({ input }): Product | undefined => getProductById(input)
  ),
});
```

- `initTRPC` sets up router helpers.
- `getAll` returns all products, type-enforced.
- `getById` validates input with Zod, returns a product or undefined.
- Uses shared `Product` type.

Connect this router to your API handler. See tRPC docs for framework details.

### Consuming tRPC API in React (`useProducts.ts`)

This code shows how to fetch products in React with full type inference.

```typescript

import { trpc } from '../trpc';

// Returns Product[] with full type safety
export function useProducts() {
  return trpc.product.getAll.useQuery();
}
```

- Imports `trpc` client.
- Calls `getAll` procedure.
- Data is typed as `Product[]`.
- TypeScript flags mismatches instantly.

### Sharing Types Between Backend and Frontend

Place shared types in a common package.

### Shared Type Example (`product.ts`)

This interface defines a product used across backend and frontend.

```typescript

export interface Product {
  id: string;
  name: string;
  price: number;
  inStock: boolean;
}
```

- Both backend and frontend import this.
- Any change triggers compile-time errors everywhere it’s used.

### Example: Product Catalog API with tRPC

Here’s a React component that fetches and displays products.

### Product List Component Using tRPC (`ProductList.tsx`)

This component fetches products and displays them, handling loading and errors.

```tsx

import { useProducts } from '../hooks/useProducts';

export function ProductList() {
  const { data, isLoading, error } = useProducts();
  if (isLoading) return <div>Loading...</div>;
  if (error) return <div>Error: {error.message}</div>;
  return (
    <ul>
      {data.map(product => (
        <li key={product.id}>
          {product.name} - ${product.price}
        </li>
      ))}
    </ul>
  );
}
```

- Fetches products with `useProducts`.
- Shows loading or error messages as needed.
- Renders a list with type-safe product data.

---

## OpenAPI Workflows for Polyglot Backends

If your backend isn’t TypeScript, use OpenAPI. It’s a universal translator—defining contracts everyone can understand.

### Generating TypeScript Clients from OpenAPI Specs

Export your OpenAPI spec (e.g., FastAPI’s `/openapi.json`). Generate TypeScript clients or types.

### Generating TypeScript Client from OpenAPI (`openapi-codegen.sh`)

This command generates a typed client from your backend spec.

```sh

npx openapi-typescript-codegen \
  --input http://localhost:8000/openapi.json \
  --output ./src/api
```

- Fetches the OpenAPI spec.
- Outputs a fully-typed client to `./src/api`.

### Consuming Generated Client in React (`fetchProducts.ts`)

This code shows how to use the generated client in React.

```typescript

import { ProductApi } from './api';

const api = new ProductApi();

async function fetchProducts() {
  // Fetch all products from the backend API
  const products = await api.getProducts();
  return products;
}
```

- Instantiates the typed API client.
- Calls `getProducts`, which returns typed data.

You can also generate only types for use with your own fetch logic.

### Validating Contracts and Error Handling

OpenAPI specs define requests, responses, and errors. Automated tools can check for breaking changes before deployment. Error responses are typed and documented.

### Managing Versioning and Backward Compatibility

Use semantic versioning in your OpenAPI spec. Mark deprecated endpoints. Automate compatibility checks in your CI pipeline to catch breaking changes early.

---

## Monorepo Architecture and Shared Types

A monorepo is a shared warehouse for types and utilities. This prevents duplication and keeps everyone in sync.

### Organizing Shared Packages and Type Definitions

Structure your monorepo for easy sharing.

### Monorepo Directory Structure Example (`structure.txt`)

```plaintext

/apps
  /frontend
  /backend
/packages
  /types
  /api-clients
```

- `/packages/types`: Shared interfaces and types.
- `/packages/api-clients`: Generated API clients.

Use pnpm workspaces for efficient dependency management.

### Automated Type Generation and CI Integration

Automate type generation and checks in your CI workflow.

### Sample CI Workflow for Type Generation (`ci-workflow.yaml`)

This workflow generates types and checks them on every build.

```yaml

jobs:
  generate-types:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - run: pnpm install
      - run: pnpm run generate:types
      - run: pnpm run typecheck
```

- Installs dependencies.
- Runs codegen and type checks.
- Fails the build if contracts drift.

### Business Impact: Reducing Integration Bugs and Support Costs

Centralized types and automated checks mean fewer bugs, faster delivery, and lower support costs.

---

## Authentication and Authorization Patterns

APIs are like VIP sections—only the right people get in. Authentication checks identity. Authorization controls access.

### Implementing OAuth, JWT, and Session Management

Modern apps use a mix of strategies:

- **OAuth:** Third-party logins (Google, Microsoft)
- **JWT:** Stateless tokens, scalable for APIs
- **Session Cookies:** Traditional web apps

#### ### JWT Authentication Flow Example (`jwt-auth.ts`)

This code issues a JWT and verifies it using secure cookies.

```typescript

// On login (backend):
import jwt from 'jsonwebtoken';
import { Response } from 'express';

function loginUser(userId: string, res: Response) {
  // Issue a JWT token valid for 1 hour
  const token = jwt.sign(
    { userId },
    process.env.JWT_SECRET,
    { expiresIn: '1h' }
  );
  // Store JWT in an HTTP-only, Secure cookie
  res.cookie('token', token, {
    httpOnly: true,
    secure: true,
    sameSite: 'strict',
    maxAge: 60 * 60 * 1000
  });
  return { success: true };
}

// On protected API request:
import { Request, Response, NextFunction } from 'express';

function authenticateRequest(
  req: Request,
  res: Response,
  next: NextFunction
) {
  const token = req.cookies.token;
  if (!token) {
    return res.status(401).json({ error: 'Authentication required' });
  }
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (err) {
    return res.status(401).json({ error: 'Invalid or expired token' });
  }
}
```

- On login, backend issues and stores a JWT.
- Requests include the token in a secure cookie.
- Middleware verifies the token and attaches user info.

**Best Practices:**
- Never store JWTs in localStorage.
- Use HTTP-only, Secure cookies.
- Enable CSRF protection.

### Integrating Authentication with Server Components and Actions

Server Components run only on the server, allowing secure checks before rendering sensitive data.

#### ### Protecting a Server Component (`UserProfile.tsx`)

This component only renders for authenticated users.

```tsx

import { getCurrentUser } from '../lib/auth';
import { redirect } from 'react-router-dom';

export default async function UserProfile() {
  const user = await getCurrentUser();
  if (!user) {
    redirect('/login');
  }
  return <div>Welcome, {user.name}!</div>;
}
```

- Fetches the user server-side.
- Redirects if unauthenticated.
- Only renders data for authenticated users.

### Securing Sensitive Routes and User Data

Use middleware to enforce authentication on API routes. Never expose sensitive data to unauthenticated users.

#### ### Express Middleware for Protected API Routes (`requireAuth.ts`)

This middleware protects API routes using JWT in cookies.

```typescript

import { Request, Response, NextFunction } from 'express';
import jwt from 'jsonwebtoken';

function requireAuth(
  req: Request,
  res: Response,
  next: NextFunction
) {
  const token = req.cookies.token;
  if (!token) {
    return res.status(401).json({ error: 'Authentication required' });
  }
  try {
    const user = jwt.verify(token, process.env.JWT_SECRET);
    req.user = user;
    next();
  } catch {
    return res.status(401).json({ error: 'Invalid or expired token' });
  }
}

// Usage:
app.get('/api/profile', requireAuth, (req, res) => {
  res.json({ user: req.user });
});
```

- Checks for JWT in cookie.
- Rejects or allows based on token validity.

### Best Practices for Enterprise Authentication

- Centralize auth logic in helpers and middleware.
- Keep secrets server-only.
- Automate security checks in CI.
- Document and regularly audit flows.

---

## Summary and Key Takeaways

Type safety is your foundation. It prevents bugs and streamlines onboarding. Use tRPC for TypeScript monorepos and OpenAPI for cross-language teams. Organize shared types in a monorepo and automate checks in CI. Secure your APIs with robust authentication and authorization.

These patterns:
- Prevent integration bugs
- Speed up development
- Reduce support costs
- Keep your data secure

For more on advanced TypeScript, see Chapter 3. For real-world case studies, see Chapter 11.

---

## Key Ideas and Glossary

| Key Idea                                      | Description                                                                 |
|-----------------------------------------------|-----------------------------------------------------------------------------|
| Type Safety                                   | Ensures data matches expected types, preventing runtime errors               |
| tRPC                                          | TypeScript-first API framework for end-to-end type safety                    |
| OpenAPI                                       | Standard for describing APIs, enabling cross-language code generation        |
| Monorepo                                      | Single repository for multiple apps and shared code                          |
| Authentication                                | Verifying user identity                                                     |
| Authorization                                | Controlling what authenticated users can do                                 |
| JWT                                           | JSON Web Token, a stateless authentication mechanism                        |
| OAuth                                         | Open Authorization protocol for third-party logins                          |
| Session Cookie                                | Stores session ID in cookie, server holds session data                      |
| Middleware                                    | Code that runs before request handlers, used for checks or preprocessing     |

---

## Exercises and Next Steps

1. **Set up a basic tRPC router and React component in a TypeScript monorepo. Share types between backend and frontend.**
   - *Hint:* Create `/packages/types`, define a router, use trpc hooks.

2. **Export an OpenAPI spec from a FastAPI backend and generate a TypeScript client. Use the client in React.**
   - *Hint:* Run `openapi-typescript-codegen`, import API methods in your app.

3. **Automate type generation and contract validation in CI. Fail the build if types are out of sync.**
   - *Hint:* Add a pnpm script for codegen and a typecheck step in CI.

4. **Implement JWT authentication in your API. Protect a React Server Component so it only renders for authenticated users.**
   - *Hint:* Issue a JWT on login, verify it server-side, and redirect unauthenticated users.

5. **Refactor an existing API integration to use shared types from a monorepo package, removing duplicate definitions.**
   - *Hint:* Move interfaces to `/packages/types` and import them everywhere.

For deeper learning, review Chapter 3 (Advanced TypeScript), Chapter 4 (Server Components), and Chapter 10 (Security Best Practices).