# Chapter 3: Advanced TypeScript for React: Building Bulletproof APIs

# Advanced TypeScript for React: Building Bulletproof APIs

## Introduction: Why TypeScript Mastery Matters for Modern React

Imagine designing a skyscraper with a pencil—mistakes slip by, and the structure is shaky. Advanced TypeScript is your laser blueprint for React, catching errors before they reach production. In 2025, React 19's server-first model and complex data flows demand strong type safety more than ever.

Why is this crucial? TypeScript prevents silent bugs, documents your architecture, and aligns teams. With `"strict": true` in your `tsconfig.json`, you enforce rigorous contracts—now a must for modern React projects.

This chapter teaches you how advanced TypeScript—generics, conditional types, mapped types, discriminated unions, the `satisfies` operator, and native execution—transforms your React APIs. You'll learn to model business logic in code, prevent misuse, and keep frontend and backend in sync.

We’ll cover:

- **Generics** for reusable, type-safe components and hooks
- **Conditional types** for dynamic props and logic
- **Mapped types** for clean, dynamic data models
- **Discriminated unions** for safe UI states
- **The `satisfies` operator** for strict prop validation
- **Native TypeScript execution** for faster, simpler workflows
- **Shared types in monorepos** for end-to-end safety

Let’s see how these patterns build bulletproof APIs for modern React.

---

## TypeScript Generics and Conditional Types

Reusable components are the core of React. Generics, conditional types, and mapped types make them flexible and safe. Think of generics as templates—write once, use with any data.

### Using Generics for Reusable Components

Generics let you write components and hooks that work with any data type, preserving type safety.

**Example: Generic List Component**

Before the code, note: This List component works for any data type. TypeScript ensures the items and render function always match.

### `List.tsx` – Generic List Component

```tsx

type ListProps<T> = {
  items: T[];
  renderItem: (item: T) => React.ReactNode;
};

function List<T>({ items, renderItem }: ListProps<T>) {
  return <ul>{items.map(renderItem)}</ul>;
}

// Usage
<List
  items={[1, 2, 3]}
  renderItem={item => <li key={item}>{item}</li>}
/>
```

- `T` is a placeholder for any type.
- TypeScript infers `T` from `items`.
- The component is reusable for any data.
- TypeScript flags mismatches instantly.

### Implementing Conditional Types for Flexible Props

Conditional types let props change shape based on a flag or value.

**Example: Button with Conditional Props**

This pattern enforces the correct props based on a boolean flag.

### `ButtonProps.ts` – Conditional Props Based on a Flag

```ts

type ButtonProps<T extends boolean> =
  T extends true
    ? { onClick: () => void; label: string }
    : { href: string; label: string };

// Usage
type Clickable = ButtonProps<true>; // Requires onClick
type Link = ButtonProps<false>;     // Requires href
```

- If `T` is `true`, `onClick` is required.
- If `T` is `false`, `href` is required.
- TypeScript enforces correct usage.

### Mapped Types for Dynamic Data Structures

Mapped types help model objects with dynamic keys—great for forms or filters.

**Example: Form State with Mapped Types**

This pattern ensures every field is present and typed.

### `FormState.ts` – Creating a Mapped Type for Form State

```ts

type FormFields = 'email' | 'password' | 'rememberMe';

type FormState = {
  [K in FormFields]: string;
};

const loginForm: FormState = {
  email: '',
  password: '',
  rememberMe: ''
};
```

- `FormFields` lists all keys.
- `FormState` enforces every field as a string.
- TypeScript flags missing or incorrect fields.

---

## Type Safety in Component and API Design

Type safety is essential for reliable React APIs. Use the `satisfies` operator, discriminated unions, and safe type narrowing to enforce contracts and prevent bugs.

### Leveraging the `satisfies` Operator for Prop Validation

The `satisfies` operator checks that an object matches a type—without widening or hiding errors.

**Example: Safe Prop Validation**

TypeScript flags typos or missing properties immediately.

### `buttonProps.ts` – Using the `satisfies` Operator for Props

```ts

const buttonProps = {
  label: 'Submit',
  onClick: () => alert('Clicked!')
} satisfies { label: string; onClick: () => void };
```

- TypeScript checks all keys.
- Typos or omissions cause compile-time errors.
- Literal values are preserved for better inference.

### Discriminated Unions for Complex UI States

Discriminated unions model mutually exclusive component states—like loading, error, or success.

**Example: Fetch State with Discriminated Unions**

This pattern ensures every state is handled explicitly.

### `FetchState.tsx` – Modeling Fetch States with Discriminated Unions

```ts

type FetchState<T> =
  | { status: 'loading' }
  | { status: 'error'; error: string }
  | { status: 'success'; data: T };

function DataDisplay<T>({ state }: { state: FetchState<T> }) {
  switch (state.status) {
    case 'loading':
      return <span>Loading...</span>;
    case 'error':
      return <span>Error: {state.error}</span>;
    case 'success':
      return <span>Data: {JSON.stringify(state.data)}</span>;
  }
}
```

- `status` discriminates each state.
- TypeScript narrows types in each case.
- Only valid properties are accessible per state.

### Avoiding Pitfalls: any, unknown, and Unsafe Casting

Avoid `any`—it disables type checks. Use `unknown` for untyped data, and always narrow before using.

**Example: Safe Type Narrowing with `unknown`**

This ensures you never trust external data blindly.

### `handleInput.ts` – Safe Type Narrowing with Unknown

```ts

function handleInput(input: unknown) {
  if (typeof input === 'string') {
    return input.toUpperCase();
  }
  throw new Error('Invalid input');
}
```

- Accepts `unknown` input.
- Checks type before use.
- Only operates on valid types.
- Throws error for invalid input.

---

## Type Inference and Native Execution

Node.js v23.6.0+ runs TypeScript files directly—no build step needed for erasable syntax. TypeScript 5.8+ offers smarter inference, reducing boilerplate.

### How Node.js v23.6.0+ Executes TypeScript Natively

Run TypeScript files directly—Node strips types at runtime.

### `RunTS.sh` – Running TypeScript Directly in Node.js

```sh

node src/server.ts
```

- No flags needed in v23.6.0+.
- Types are stripped; code runs as JavaScript.
- For older Node.js, use `--experimental-strip-types`.

### Checking for Node.js-Compatible TypeScript Syntax

Verify your code uses only erasable syntax.

### `CheckSyntax.sh` – Type-Checking for Node.js-Compatible TypeScript

```sh

tsc --noEmit --erasableSyntaxOnly
```

- Ensures only erasable syntax is used.
- Catches type errors before runtime.

### Type Inference with Generics

TypeScript infers types for you, making code cleaner.

### `mapValues.ts` – Type Inference with Generics

```ts

function mapValues<T, U>(
  arr: T[],
  fn: (item: T) => U
) {
  return arr.map(fn);
}

const result = mapValues([1, 2, 3], x => x.toString());
// result is inferred as string[]
```

- TypeScript infers `T` and `U`.
- No need for manual annotations.
- Result is correctly typed.

---

## Shared Types in Monorepos: Patterns and Practice

Shared types keep frontend and backend in sync. Organize them in a dedicated package for easy imports and updates.

### Organizing and Sharing Type Definitions

Structure your monorepo for shared type safety.

### `monorepo-structure.txt` – Monorepo Structure for Shared Types

```text

apps/
  frontend/
  backend/
packages/
  types/
    product.ts
    user.ts
```

- Both frontend and backend import from `types`.
- Changes to types propagate everywhere.

### Defining a Product Type for API Contracts

Define types once and use them across your stack.

### `product.ts` – Defining a Product Type for API Contracts

```ts

export type Product = {
  id: string;
  name: string;
  price: number;
};
```

- `Product` has `id`, `name`, and `price`.
- Used in API, database, and UI.

### Server Fetch and Client Render with Shared Types

Use shared types for both backend and frontend logic.

### `getProducts.ts` and `ProductList.tsx` – Server Fetch and Client Render

```ts

// Backend API
import { Product } from '@acme/types/product';

export async function getProducts(): Promise<Product[]> {
  // ...fetch from DB
}

// Frontend usage
import { Product } from '@acme/types/product';

function ProductList({ products }: { products: Product[] }) {
  return (
    <ul>
      {products.map((p) => (
        <li key={p.id}>{p.name}: ${p.price}</li>
      ))}
    </ul>
  );
}
```

- Backend and frontend use the same `Product` type.
- Changes to `Product` are enforced everywhere.
- Prevents silent contract mismatches.

---

## Summary and Key Takeaways

This chapter gave you advanced TypeScript tools for robust, maintainable React APIs. You learned:

- **Generics** for reusable, type-safe components and hooks
- **Conditional types** for dynamic props and logic
- **Mapped types** for modeling dynamic data
- **The `satisfies` operator** for strict prop validation
- **Discriminated unions** for safe, clear UI states
- **Strict mode** for maximum type safety
- **Native TypeScript execution** for faster workflows
- **Shared types in monorepos** for end-to-end safety

These patterns prevent bugs, speed up onboarding, and help teams scale. Master them to build future-proof React apps.

---

### Key Ideas

- Use generics, conditional types, mapped types, and utility types for reusable, safe components.
- The `satisfies` operator and discriminated unions make APIs safer and more readable.
- Always enable strict mode (`"strict": true`) in TypeScript.
- Avoid `any` and unsafe casts—prefer `unknown` and type narrowing.
- Explicitly type React events and server actions.
- Node.js v23.6.0+ enables direct TypeScript execution.
- Shared types in monorepos keep your stack in sync.

---

### Glossary

- **Generic:** Code that works with any data type while staying type-safe.
- **Conditional Type:** A type that changes based on another type or value.
- **Mapped Type:** Creates new types by transforming keys from another type.
- **Utility Type:** Built-in helpers like `Partial<T>`, `Pick<T, K>`, `Record<K, T>`.
- **Discriminated Union:** Union of object types with a shared property for safe branching.
- **Template Literal Type:** Composes string literal types using template syntax.
- **`satisfies` Operator:** Checks if a value matches a type, preserving its details.
- **`as const`:** Marks values as immutable and narrows their types to exact literals.
- **Native TypeScript Execution:** Running `.ts` files directly in Node.js.
- **Monorepo:** One repository with multiple projects and shared code.

---

## Exercises and Next Steps

**1. Refactor a React component (like a Table or List) to use generics for type-safe rendering of any data type.**

*Hint:* Add a generic type parameter to your props and use it for both items and render function.

**2. Create a discriminated union to model loading, error, and success states for a data-fetching component. Render different UI for each state.**

*Hint:* Use a `status` field and a switch statement to handle each case.

**3. Use the `satisfies` operator to validate an object against a prop type. Intentionally introduce a typo and observe the type error.**

*Hint:* Misspell or remove a required property to see TypeScript's error.

**4. Organize a shared type (e.g., Product or User) in a simulated monorepo. Import and use it in both backend and frontend code.**

*Hint:* Create a `types` folder or package, export your type, and import in both places.

**5. Run a TypeScript file directly in Node.js v23.6.0+ using `node yourfile.ts`. Compare this to traditional compilation.**

*Hint:* Write a simple `.ts` file and execute it with `node`.

---

*For deeper dives into API integration and monorepo patterns, see Chapter 7: Full-Stack Type Safety and API Integration.*